<?php

namespace App\Observers;

use App\Models\Service;
use App\Models\Commission;
use App\Models\Completion;
use Carbon\Carbon;

class ServiceObserver
{
    /**
     * Handle the Service "updated" event.
     * Recalculate commissions when service commission rate changes.
     */
    public function updated(Service $service): void
    {
        // Check if commission rate changed
        if ($service->isDirty('commission')) {
            $oldCommission = $service->getOriginal('commission');
            $newCommission = $service->commission;

            if ($oldCommission != $newCommission) {
                $this->recalculateCommissions($service);
            }
        }
    }

    /**
     * Handle the Service "creating" event.
     * Adjust timestamps with timezone (+7 hours).
     */
    public function creating(Service $service): void
    {
        if ($service->created_at) {
            $service->created_at = Carbon::parse($service->created_at)->addHours(7);
        }
    }

    /**
     * Recalculate all commissions for completions of this service.
     */
    private function recalculateCommissions(Service $service): void
    {
        // Get all completions for this service
        $completions = Completion::where('service_id', $service->service_id)->get();

        foreach ($completions as $completion) {
            // Find and update commissions for this completion
            $commissions = Commission::where('completions_id', $completion->completions_id)->get();

            foreach ($commissions as $commission) {
                $commission->commission_amount = ($completion->charged * $service->commission) / 100;
                $commission->user_id = $completion->user_id;
                $commission->updated_at = Carbon::parse($completion->date)->addHours(7);
                $commission->save();
            }
        }
    }
}
