<?php

namespace App\Observers;

use App\Models\Completion;
use App\Models\Commission;
use Carbon\Carbon;

class CompletionObserver
{
    /**
     * Handle the Completion "creating" event.
     * Adjust timestamps with timezone (+7 hours).
     */
    public function creating(Completion $completion): void
    {
        if ($completion->created_at) {
            $completion->created_at = Carbon::parse($completion->created_at)->addHours(7);
        }

        if ($completion->date) {
            $completion->date = Carbon::parse($completion->date)->addHours(7);
        }
    }

    /**
     * Handle the Completion "created" event.
     * Create commission if the completion is approved.
     */
    public function created(Completion $completion): void
    {
        // If completion is approved, create a commission
        if ($completion->approved && $completion->charged > 0) {
            $service = $completion->service;

            if ($service) {
                Commission::create([
                    'completions_id' => $completion->completions_id,
                    'user_id' => $completion->user_id,
                    'commission_amount' => ($completion->charged * $service->commission) / 100,
                ]);
            }
        }
    }

    /**
     * Handle the Completion "updated" event.
     * Automatically manages commissions based on ANY changes to the completion:
     * - If approved changes to false (0): Delete all related commissions
     * - If approved changes to true (1): Create new commission if needed
     * - If charged amount changes while approved: Update commission amount
     * - If service changes while approved: Recalculate commission
     */
    public function updated(Completion $completion): void
    {
        // Case 1: Approved status changed to FALSE - DELETE commissions
        if ($completion->isDirty('approved') && !$completion->approved) {
            Commission::where('completions_id', $completion->completions_id)->delete();
            return;
        }

        // Case 2: Approved status changed to TRUE - CREATE commission if not exists
        if ($completion->isDirty('approved') && $completion->approved) {
            // Reload service relationship to get fresh data
            $service = $completion->service()->first();

            if (!$service) {
                return;
            }

            // Check if commission already exists
            $existingCommission = Commission::where('completions_id', $completion->completions_id)->first();

            if (!$existingCommission && $completion->charged > 0) {
                Commission::create([
                    'completions_id' => $completion->completions_id,
                    'user_id' => $completion->user_id,
                    'commission_amount' => ($completion->charged * $service->commission) / 100,
                ]);
            }
            return;
        }

        // Case 3: Completion is approved and charged/service changed - UPDATE commission
        if ($completion->approved && ($completion->isDirty('charged') || $completion->isDirty('service_id'))) {
            // Reload service relationship to get fresh data
            $service = $completion->service()->first();

            if (!$service) {
                return;
            }

            $commission = Commission::where('completions_id', $completion->completions_id)->first();

            if ($commission) {
                // Recalculate commission amount
                $newAmount = ($completion->charged * $service->commission) / 100;
                $commission->update([
                    'commission_amount' => $newAmount,
                    'user_id' => $completion->user_id, // Update user if changed
                ]);
            } elseif ($completion->charged > 0) {
                // Create commission if it doesn't exist but should
                Commission::create([
                    'completions_id' => $completion->completions_id,
                    'user_id' => $completion->user_id,
                    'commission_amount' => ($completion->charged * $service->commission) / 100,
                ]);
            }
        }
    }

    /**
     * Handle the Completion "deleting" event.
     * Delete related commissions before the completion is deleted.
     */
    public function deleting(Completion $completion): void
    {
        // Delete all related commissions
        Commission::where('completions_id', $completion->completions_id)->delete();
    }
}
