<?php

namespace App\Observers;

use App\Models\Commission;
use App\Models\Completion;
use Carbon\Carbon;

class CommissionObserver
{
    /**
     * Handle the Commission "creating" event.
     * Calculate commission amount automatically before creating.
     */
    public function creating(Commission $commission): void
    {
        $this->calculateCommission($commission);
    }

    /**
     * Handle the Commission "updating" event.
     * Recalculate commission amount before updating.
     */
    public function updating(Commission $commission): void
    {
        $this->calculateCommission($commission);
    }

    /**
     * Calculate commission amount based on completion and service data.
     */
    private function calculateCommission(Commission $commission): void
    {
        // Get the completion with service
        $completion = Completion::with('service')->find($commission->completions_id);

        if (!$completion) {
            return;
        }

        // Calculate commission amount
        $rate = $completion->service->commission ?? 0;
        $charged = $completion->charged ?? 0;
        $commission->commission_amount = ($charged * $rate) / 100;

        // Set user_id from completion if not already set
        if (!$commission->user_id) {
            $commission->user_id = $completion->user_id;
        }

        // Set timestamps with timezone adjustment (+7 hours)
        $adjustedDate = Carbon::parse($completion->date)->addHours(7);

        if (!$commission->exists) {
            // Creating new record
            $commission->created_at = $adjustedDate;
            $commission->updated_at = $adjustedDate;
        } else {
            // Updating existing record
            $commission->updated_at = $adjustedDate;
        }
    }
}
