<?php

namespace App\Http\Controllers;

use App\Models\Sale;
use App\Models\SaleItem;
use App\Models\InventoryItem;
use App\Models\StockMovement;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class SalesController extends Controller
{
    /**
     * Display a listing of sales.
     */
    public function index(Request $request)
    {
        $query = Sale::with(['user', 'customer', 'saleItems']);

        // Filter by date range
        if ($request->has('start_date') && $request->has('end_date')) {
            $query->whereBetween('sale_date', [$request->start_date, $request->end_date]);
        }

        // Filter by payment status
        if ($request->has('payment_status')) {
            $query->where('payment_status', $request->payment_status);
        }

        // Filter by payment method
        if ($request->has('payment_method')) {
            $query->where('payment_method', $request->payment_method);
        }

        // Filter by user (cashier)
        if ($request->has('user_id')) {
            $query->where('user_id', $request->user_id);
        }

        // Search
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('sale_number', 'like', "%{$search}%")
                    ->orWhere('receipt_number', 'like', "%{$search}%");
            });
        }

        $sales = $query->orderBy('sale_date', 'desc')
            ->paginate($request->get('per_page', 15));

        return response()->json($sales);
    }

    /**
     * Store a newly created sale.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'customer_id' => 'nullable|exists:users,id',
            'items' => 'required|array|min:1',
            'items.*.inventory_item_id' => 'required|exists:inventory_items,id',
            'items.*.quantity' => 'required|numeric|min:0.01',
            'items.*.discount' => 'nullable|numeric|min:0',
            'tax_amount' => 'nullable|numeric|min:0',
            'discount_amount' => 'nullable|numeric|min:0',
            'payment_method' => 'required|in:cash,card,mobile_money,bank_transfer,credit',
            'amount_paid' => 'required|numeric|min:0',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        DB::beginTransaction();
        try {
            $subtotal = 0;
            $saleItemsData = [];

            // Validate stock and calculate totals
            foreach ($request->items as $item) {
                $inventoryItem = InventoryItem::findOrFail($item['inventory_item_id']);

                // Check stock availability
                if ($inventoryItem->quantity < $item['quantity']) {
                    DB::rollBack();
                    return response()->json([
                        'message' => "Insufficient stock for {$inventoryItem->name}. Available: {$inventoryItem->quantity}"
                    ], 400);
                }

                $itemDiscount = $item['discount'] ?? 0;
                $itemSubtotal = $inventoryItem->selling_price * $item['quantity'];
                $itemTotal = $itemSubtotal - $itemDiscount;

                $subtotal += $itemSubtotal;

                $saleItemsData[] = [
                    'inventory_item_id' => $inventoryItem->id,
                    'item_name' => $inventoryItem->name,
                    'sku' => $inventoryItem->sku,
                    'quantity' => $item['quantity'],
                    'unit_price' => $inventoryItem->selling_price,
                    'buying_price' => $inventoryItem->buying_price,
                    'subtotal' => $itemSubtotal,
                    'discount' => $itemDiscount,
                    'total' => $itemTotal,
                ];
            }

            $taxAmount = $request->tax_amount ?? 0;
            $discountAmount = $request->discount_amount ?? 0;
            $totalAmount = $subtotal + $taxAmount - $discountAmount;
            $changeAmount = max(0, $request->amount_paid - $totalAmount);

            // Create sale
            $sale = Sale::create([
                'user_id' => Auth::id() ?? 1, // Default to admin if not authenticated
                'customer_id' => $request->customer_id,
                'subtotal' => $subtotal,
                'tax_amount' => $taxAmount,
                'discount_amount' => $discountAmount,
                'total_amount' => $totalAmount,
                'payment_method' => $request->payment_method,
                'amount_paid' => $request->amount_paid,
                'change_amount' => $changeAmount,
                'payment_status' => $request->amount_paid >= $totalAmount ? 'completed' : 'pending',
                'notes' => $request->notes,
                'sale_date' => now(),
            ]);

            // Create sale items and update inventory
            foreach ($saleItemsData as $saleItemData) {
                $inventoryItem = InventoryItem::find($saleItemData['inventory_item_id']);
                $previousQuantity = $inventoryItem->quantity;

                // Create sale item
                $saleItem = $sale->saleItems()->create($saleItemData);

                // Update inventory
                $inventoryItem->quantity -= $saleItemData['quantity'];
                $inventoryItem->total_sold += $saleItemData['quantity'];
                $inventoryItem->total_revenue += $saleItemData['total'];
                $inventoryItem->save();

                // Record stock movement
                StockMovement::create([
                    'inventory_item_id' => $inventoryItem->id,
                    'user_id' => Auth::id() ?? 1, // Default to admin if not authenticated
                    'sale_id' => $sale->id,
                    'type' => 'out',
                    'quantity' => $saleItemData['quantity'],
                    'previous_quantity' => $previousQuantity,
                    'new_quantity' => $inventoryItem->quantity,
                    'reason' => 'Sale',
                    'reference_number' => $sale->sale_number,
                ]);
            }

            DB::commit();

            return response()->json([
                'message' => 'Sale completed successfully',
                'data' => $sale->load(['saleItems', 'user', 'customer'])
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'Failed to complete sale',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified sale.
     */
    public function show($id)
    {
        $sale = Sale::with(['user', 'customer', 'saleItems.inventoryItem'])->findOrFail($id);
        return response()->json($sale);
    }

    /**
     * Get sales analytics.
     */
    public function analytics(Request $request)
    {
        $period = $request->get('period', 'month'); // day, week, month, custom
        $startDate = $request->start_date ?? now()->startOfMonth();
        $endDate = $request->end_date ?? now()->endOfMonth();

        switch ($period) {
            case 'day':
                $startDate = now()->startOfDay();
                $endDate = now()->endOfDay();
                break;
            case 'week':
                $startDate = now()->startOfWeek();
                $endDate = now()->endOfWeek();
                break;
            case 'month':
                $startDate = now()->startOfMonth();
                $endDate = now()->endOfMonth();
                break;
        }

        $sales = Sale::whereBetween('sale_date', [$startDate, $endDate])
            ->where('payment_status', 'completed')
            ->get();

        $totalSales = $sales->count();
        $totalRevenue = $sales->sum('total_amount');
        $totalProfit = $sales->sum(function ($sale) {
            return $sale->getTotalProfit();
        });

        // Get sales by payment method
        $salesByPaymentMethod = Sale::whereBetween('sale_date', [$startDate, $endDate])
            ->where('payment_status', 'completed')
            ->select('payment_method', DB::raw('COUNT(*) as count'), DB::raw('SUM(total_amount) as total'))
            ->groupBy('payment_method')
            ->get();

        // Get top selling items
        $topSellingItems = SaleItem::whereHas('sale', function ($query) use ($startDate, $endDate) {
            $query->whereBetween('sale_date', [$startDate, $endDate])
                ->where('payment_status', 'completed');
        })
            ->select('inventory_item_id', 'item_name', DB::raw('SUM(quantity) as total_quantity'), DB::raw('SUM(total) as total_revenue'))
            ->groupBy('inventory_item_id', 'item_name')
            ->orderBy('total_quantity', 'desc')
            ->limit(10)
            ->get();

        // Daily sales trend
        $dailySales = Sale::whereBetween('sale_date', [$startDate, $endDate])
            ->where('payment_status', 'completed')
            ->select(DB::raw('DATE(sale_date) as date'), DB::raw('COUNT(*) as count'), DB::raw('SUM(total_amount) as total'))
            ->groupBy(DB::raw('DATE(sale_date)'))
            ->orderBy('date')
            ->get();

        return response()->json([
            'period' => $period,
            'start_date' => $startDate,
            'end_date' => $endDate,
            'total_sales' => $totalSales,
            'total_revenue' => round($totalRevenue, 2),
            'total_profit' => round($totalProfit, 2),
            'average_sale' => $totalSales > 0 ? round($totalRevenue / $totalSales, 2) : 0,
            'sales_by_payment_method' => $salesByPaymentMethod,
            'top_selling_items' => $topSellingItems,
            'daily_sales' => $dailySales,
        ]);
    }

    /**
     * Get sales summary.
     */
    public function summary()
    {
        $today = Sale::whereDate('sale_date', today())
            ->where('payment_status', 'completed')
            ->sum('total_amount');

        $week = Sale::whereBetween('sale_date', [now()->startOfWeek(), now()->endOfWeek()])
            ->where('payment_status', 'completed')
            ->sum('total_amount');

        $month = Sale::whereBetween('sale_date', [now()->startOfMonth(), now()->endOfMonth()])
            ->where('payment_status', 'completed')
            ->sum('total_amount');

        $todayCount = Sale::whereDate('sale_date', today())
            ->where('payment_status', 'completed')
            ->count();

        $weekCount = Sale::whereBetween('sale_date', [now()->startOfWeek(), now()->endOfWeek()])
            ->where('payment_status', 'completed')
            ->count();

        $monthCount = Sale::whereBetween('sale_date', [now()->startOfMonth(), now()->endOfMonth()])
            ->where('payment_status', 'completed')
            ->count();

        return response()->json([
            'today' => [
                'revenue' => round($today, 2),
                'count' => $todayCount,
            ],
            'week' => [
                'revenue' => round($week, 2),
                'count' => $weekCount,
            ],
            'month' => [
                'revenue' => round($month, 2),
                'count' => $monthCount,
            ],
        ]);
    }

    /**
     * Refund a sale.
     */
    public function refund(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'reason' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $sale = Sale::with('saleItems')->findOrFail($id);

        if ($sale->payment_status === 'refunded') {
            return response()->json(['message' => 'Sale already refunded'], 400);
        }

        DB::beginTransaction();
        try {
            // Restore inventory
            foreach ($sale->saleItems as $saleItem) {
                $inventoryItem = $saleItem->inventoryItem;
                $previousQuantity = $inventoryItem->quantity;

                $inventoryItem->quantity += $saleItem->quantity;
                $inventoryItem->total_sold -= $saleItem->quantity;
                $inventoryItem->total_revenue -= $saleItem->total;
                $inventoryItem->save();

                // Record stock movement
                StockMovement::create([
                    'inventory_item_id' => $inventoryItem->id,
                    'user_id' => Auth::id() ?? 1, // Default to admin if not authenticated
                    'sale_id' => $sale->id,
                    'type' => 'return',
                    'quantity' => $saleItem->quantity,
                    'previous_quantity' => $previousQuantity,
                    'new_quantity' => $inventoryItem->quantity,
                    'reason' => 'Refund: ' . $request->reason,
                    'reference_number' => $sale->sale_number,
                ]);
            }

            // Update sale status
            $sale->payment_status = 'refunded';
            $sale->notes = ($sale->notes ? $sale->notes . "\n\n" : '') . "REFUNDED: " . $request->reason;
            $sale->save();

            DB::commit();

            return response()->json([
                'message' => 'Sale refunded successfully',
                'data' => $sale
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'Failed to refund sale',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
