<?php

namespace App\Http\Controllers;

use App\Models\InventoryItem;
use App\Models\StockMovement;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class InventoryController extends Controller
{
    /**
     * Display a listing of inventory items.
     */
    public function index(Request $request)
    {
        $query = InventoryItem::with('supplier');

        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Filter by category
        if ($request->has('category')) {
            $query->where('category', $request->category);
        }

        // Filter by supplier
        if ($request->has('supplier_id')) {
            $query->where('supplier_id', $request->supplier_id);
        }

        // Filter low stock
        if ($request->has('low_stock') && $request->low_stock) {
            $query->lowStock();
        }

        // Filter out of stock
        if ($request->has('out_of_stock') && $request->out_of_stock) {
            $query->outOfStock();
        }

        // Search
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('sku', 'like', "%{$search}%")
                    ->orWhere('barcode', 'like', "%{$search}%")
                    ->orWhere('description', 'like', "%{$search}%");
            });
        }

        $items = $query->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 15));

        return response()->json($items);
    }

    /**
     * Store a newly created inventory item.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'sku' => 'required|string|unique:inventory_items,sku',
            'description' => 'nullable|string',
            'category' => 'nullable|string|max:255',
            'unit' => 'required|string|max:50',
            'supplier_id' => 'nullable|exists:suppliers,id',
            'quantity' => 'required|numeric|min:0',
            'reorder_level' => 'required|numeric|min:0',
            'minimum_stock' => 'nullable|numeric|min:0',
            'maximum_stock' => 'nullable|numeric|min:0',
            'buying_price' => 'required|numeric|min:0',
            'selling_price' => 'required|numeric|min:0',
            'markup_percentage' => 'nullable|numeric|min:0',
            'barcode' => 'nullable|string|unique:inventory_items,barcode',
            'location' => 'nullable|string|max:255',
            'expiry_date' => 'nullable|date',
            'image_url' => 'nullable|string',
            'status' => 'in:active,inactive,discontinued',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        DB::beginTransaction();
        try {
            $item = InventoryItem::create(array_merge(
                $request->all(),
                ['last_restocked_at' => now()]
            ));

            // Record initial stock movement
            StockMovement::create([
                'inventory_item_id' => $item->id,
                'user_id' => Auth::id() ?? 1, // Default to user_id 1 if not authenticated
                'type' => 'in',
                'quantity' => $item->quantity,
                'previous_quantity' => 0,
                'new_quantity' => $item->quantity,
                'reason' => 'Initial stock',
            ]);

            DB::commit();

            return response()->json([
                'message' => 'Inventory item created successfully',
                'data' => $item->load('supplier')
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to create inventory item: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'data' => $request->all()
            ]);
            return response()->json(['message' => 'Failed to create item', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Display the specified inventory item.
     */
    public function show($id)
    {
        $item = InventoryItem::with(['supplier', 'stockMovements' => function ($query) {
            $query->latest()->limit(10);
        }])->findOrFail($id);

        return response()->json($item);
    }

    /**
     * Update the specified inventory item.
     */
    public function update(Request $request, $id)
    {
        $item = InventoryItem::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'name' => 'string|max:255',
            'sku' => 'string|unique:inventory_items,sku,' . $id,
            'description' => 'nullable|string',
            'category' => 'nullable|string|max:255',
            'unit' => 'string|max:50',
            'supplier_id' => 'nullable|exists:suppliers,id',
            'reorder_level' => 'numeric|min:0',
            'minimum_stock' => 'numeric|min:0',
            'maximum_stock' => 'nullable|numeric|min:0',
            'buying_price' => 'numeric|min:0',
            'selling_price' => 'numeric|min:0',
            'markup_percentage' => 'nullable|numeric|min:0',
            'barcode' => 'nullable|string|unique:inventory_items,barcode,' . $id,
            'location' => 'nullable|string|max:255',
            'expiry_date' => 'nullable|date',
            'image_url' => 'nullable|string',
            'status' => 'in:active,inactive,discontinued',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $item->update($request->all());

        return response()->json([
            'message' => 'Inventory item updated successfully',
            'data' => $item->load('supplier')
        ]);
    }

    /**
     * Remove the specified inventory item.
     */
    public function destroy($id)
    {
        $item = InventoryItem::findOrFail($id);
        $item->delete();

        return response()->json([
            'message' => 'Inventory item deleted successfully'
        ]);
    }

    /**
     * Update stock quantity.
     */
    public function updateStock(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'quantity' => 'required|numeric',
            'type' => 'required|in:in,out,adjustment,return,damage,expired',
            'reason' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $item = InventoryItem::findOrFail($id);
        $previousQuantity = $item->quantity;

        DB::beginTransaction();
        try {
            // Calculate new quantity
            if ($request->type === 'in' || $request->type === 'return') {
                $newQuantity = $previousQuantity + $request->quantity;
            } else {
                $newQuantity = $previousQuantity - $request->quantity;
            }

            // Update item quantity
            $item->quantity = max(0, $newQuantity);
            if ($request->type === 'in') {
                $item->last_restocked_at = now();
            }
            $item->save();

            // Record stock movement
            StockMovement::create([
                'inventory_item_id' => $item->id,
                'user_id' => Auth::id() ?? 1, // Default to user_id 1 if not authenticated
                'type' => $request->type,
                'quantity' => $request->quantity,
                'previous_quantity' => $previousQuantity,
                'new_quantity' => $item->quantity,
                'reason' => $request->reason,
                'reference_number' => $request->reference_number,
            ]);

            DB::commit();

            return response()->json([
                'message' => 'Stock updated successfully',
                'data' => $item
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Failed to update stock', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Get inventory alerts.
     */
    public function alerts()
    {
        $lowStock = InventoryItem::active()->lowStock()->with('supplier')->get();
        $outOfStock = InventoryItem::active()->outOfStock()->with('supplier')->get();

        // Get items with low sales (less than 5 units sold in the last 30 days)
        $lowSalesThreshold = 5;
        $lowSales = InventoryItem::active()
            ->whereHas('saleItems', function ($query) {
                $query->where('created_at', '>=', now()->subDays(30));
            }, '=', 0)
            ->orWhereHas('saleItems', function ($query) use ($lowSalesThreshold) {
                $query->where('created_at', '>=', now()->subDays(30))
                    ->havingRaw('SUM(quantity) < ?', [$lowSalesThreshold]);
            })
            ->with('supplier')
            ->get();

        // Get expiring items
        $expiring = InventoryItem::active()
            ->where('expiry_date', '!=', null)
            ->where('expiry_date', '<=', now()->addDays(30))
            ->where('expiry_date', '>=', now())
            ->with('supplier')
            ->get();

        // Get expired items
        $expired = InventoryItem::active()
            ->where('expiry_date', '!=', null)
            ->where('expiry_date', '<', now())
            ->with('supplier')
            ->get();

        return response()->json([
            'low_stock' => $lowStock,
            'out_of_stock' => $outOfStock,
            'low_sales' => $lowSales,
            'expiring_soon' => $expiring,
            'expired' => $expired,
        ]);
    }

    /**
     * Get inventory statistics.
     */
    public function statistics()
    {
        $totalItems = InventoryItem::active()->count();
        $totalValue = InventoryItem::active()->sum(DB::raw('quantity * buying_price'));
        $totalRetailValue = InventoryItem::active()->sum(DB::raw('quantity * selling_price'));
        $lowStockCount = InventoryItem::active()->lowStock()->count();
        $outOfStockCount = InventoryItem::active()->outOfStock()->count();

        return response()->json([
            'total_items' => $totalItems,
            'total_value' => round($totalValue, 2),
            'total_retail_value' => round($totalRetailValue, 2),
            'potential_profit' => round($totalRetailValue - $totalValue, 2),
            'low_stock_count' => $lowStockCount,
            'out_of_stock_count' => $outOfStockCount,
        ]);
    }

    /**
     * Get categories.
     */
    public function categories()
    {
        $categories = DB::table('categories')
            ->orderBy('name')
            ->get(['id', 'name', 'description', 'icon', 'color', 'status']);

        return response()->json($categories);
    }

    /**
     * Create a new category.
     */
    public function storeCategory(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:100|unique:categories,name',
            'description' => 'nullable|string|max:500',
            'icon' => 'nullable|string|max:50',
            'color' => 'nullable|string|max:7',
            'status' => 'in:active,inactive',
        ]);

        $categoryId = DB::table('categories')->insertGetId([
            'name' => $validated['name'],
            'description' => $validated['description'] ?? null,
            'icon' => $validated['icon'] ?? null,
            'color' => $validated['color'] ?? null,
            'status' => $validated['status'] ?? 'active',
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        $category = DB::table('categories')->where('id', $categoryId)->first();

        return response()->json([
            'message' => 'Category created successfully',
            'data' => $category
        ], 201);
    }

    /**
     * Update a category.
     */
    public function updateCategory(Request $request, $id)
    {
        $category = DB::table('categories')->where('id', $id)->first();

        if (!$category) {
            return response()->json(['message' => 'Category not found'], 404);
        }

        $validated = $request->validate([
            'name' => 'sometimes|required|string|max:100|unique:categories,name,' . $id,
            'description' => 'nullable|string|max:500',
            'icon' => 'nullable|string|max:50',
            'color' => 'nullable|string|max:7',
            'status' => 'in:active,inactive',
        ]);

        DB::table('categories')
            ->where('id', $id)
            ->update([
                'name' => $validated['name'] ?? $category->name,
                'description' => $validated['description'] ?? $category->description,
                'icon' => $validated['icon'] ?? $category->icon,
                'color' => $validated['color'] ?? $category->color,
                'status' => $validated['status'] ?? $category->status,
                'updated_at' => now(),
            ]);

        $updatedCategory = DB::table('categories')->where('id', $id)->first();

        return response()->json([
            'message' => 'Category updated successfully',
            'data' => $updatedCategory
        ]);
    }

    /**
     * Delete a category.
     */
    public function destroyCategory($id)
    {
        $category = DB::table('categories')->where('id', $id)->first();

        if (!$category) {
            return response()->json(['message' => 'Category not found'], 404);
        }

        // Check if category is in use
        $itemsCount = InventoryItem::where('category', $category->name)->count();

        if ($itemsCount > 0) {
            return response()->json([
                'message' => "Cannot delete category. It is being used by {$itemsCount} inventory item(s)."
            ], 422);
        }

        DB::table('categories')->where('id', $id)->delete();

        return response()->json(['message' => 'Category deleted successfully']);
    }
}
