<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Signin;
use Illuminate\Http\Request;

class SigninController extends Controller
{
    public function index(Request $request)
    {
        $query = Signin::with('user');

        if ($request->has('user_id')) {
            $query->where('user_id', $request->user_id);
        }

        if ($request->has('start_date')) {
            $query->whereDate('created_at', '>=', $request->start_date);
        }
        if ($request->has('end_date')) {
            $query->whereDate('created_at', '<=', $request->end_date);
        }

        $signins = $query->orderByDesc('created_at')->get();

        return response()->json([
            'success' => true,
            'data' => $signins,
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'user_id' => 'required|exists:users,user_id',
            'status' => 'required|boolean',
            'approved' => 'nullable|boolean',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
            'location_address' => 'nullable|string|max:500',
        ]);

        $signin = Signin::create($validated);

        return response()->json([
            'success' => true,
            'message' => 'Signin recorded successfully',
            'data' => $signin->load('user'),
        ], 201);
    }

    public function show($id)
    {
        $signin = Signin::with('user')->findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => $signin,
        ]);
    }

    public function update(Request $request, $id)
    {
        $signin = Signin::findOrFail($id);

        $validated = $request->validate([
            'approved' => 'sometimes|boolean',
            'status' => 'sometimes|boolean',
        ]);

        $signin->update($validated);

        return response()->json([
            'success' => true,
            'message' => 'Signin updated successfully',
            'data' => $signin->load('user'),
        ]);
    }

    public function destroy($id)
    {
        $signin = Signin::findOrFail($id);
        $signin->delete();

        return response()->json([
            'success' => true,
            'message' => 'Signin deleted successfully',
        ]);
    }
}
