<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Completion;
use App\Models\Commission;
use Illuminate\Http\Request;

class CompletionController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Completion::with(['service', 'user', 'commission']);

        // Filter by user
        if ($request->has('user_id')) {
            $query->where('user_id', $request->user_id);
        }

        // Filter by service
        if ($request->has('service_id')) {
            $query->where('service_id', $request->service_id);
        }

        // Filter by approved status
        if ($request->has('approved')) {
            $query->where('approved', filter_var($request->approved, FILTER_VALIDATE_BOOLEAN));
        }

        // Filter by date range
        if ($request->has('start_date')) {
            $query->whereDate('date', '>=', $request->start_date);
        }
        if ($request->has('end_date')) {
            $query->whereDate('date', '<=', $request->end_date);
        }

        $completions = $query->orderByDesc('date')->get();

        return response()->json([
            'success' => true,
            'data' => $completions,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'user_id' => 'required|exists:users,user_id',
            'service_id' => 'required|exists:services,service_id',
            'charged' => 'required|numeric|min:0',
            'date' => 'required|date',
            'approved' => 'sometimes|boolean',
        ]);

        $completion = Completion::create($validated);

        // If approved, create commission record (observer will calculate amount)
        if ($validated['approved'] ?? false) {
            Commission::create([
                'completions_id' => $completion->completions_id,
                // user_id and commission_amount will be set by CommissionObserver
            ]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Completion created successfully',
            'data' => $completion->load(['service', 'user', 'commission']),
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $completion = Completion::with(['service', 'user', 'commission'])->findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => $completion,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $completion = Completion::findOrFail($id);

        $validated = $request->validate([
            'user_id' => 'sometimes|exists:users,user_id',
            'service_id' => 'sometimes|exists:services,service_id',
            'charged' => 'sometimes|numeric|min:0',
            'date' => 'sometimes|date',
            'approved' => 'sometimes|boolean',
        ]);

        // Track if approval status is changing
        $wasApproved = (bool) $completion->approved;
        $willBeApproved = $validated['approved'] ?? $wasApproved;

        $completion->update($validated);

        // If completion was just approved, create commission record
        // Observer will automatically delete commission if approval changes to false
        if (!$wasApproved && $willBeApproved) {
            // Check if commission doesn't already exist
            $existingCommission = Commission::where('completions_id', $completion->completions_id)->first();

            if (!$existingCommission) {
                Commission::create([
                    'completions_id' => $completion->completions_id,
                    // user_id and commission_amount will be set by CommissionObserver
                ]);
            }
        }

        return response()->json([
            'success' => true,
            'message' => 'Completion updated successfully',
            'data' => $completion->load(['service', 'user', 'commission']),
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $completion = Completion::findOrFail($id);
        $completion->delete();

        return response()->json([
            'success' => true,
            'message' => 'Completion deleted successfully',
        ]);
    }
}
