<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Commission;
use Illuminate\Http\Request;

class CommissionController extends Controller
{
    public function index(Request $request)
    {
        $query = Commission::with(['user', 'completion.service']);

        if ($request->has('user_id')) {
            $query->where('user_id', $request->user_id);
        }

        if ($request->has('start_date')) {
            $query->whereDate('created_at', '>=', $request->start_date);
        }
        if ($request->has('end_date')) {
            $query->whereDate('created_at', '<=', $request->end_date);
        }

        $commissions = $query->orderByDesc('created_at')->get();

        return response()->json([
            'success' => true,
            'data' => $commissions,
            'total' => $commissions->sum('commission_amount'),
        ])->header('Cache-Control', 'no-cache, no-store, must-revalidate')
            ->header('Pragma', 'no-cache')
            ->header('Expires', '0');
    }

    public function show($id)
    {
        $commission = Commission::with(['user', 'completion.service'])->findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => $commission,
        ]);
    }

    public function summary(Request $request)
    {
        $userId = $request->query('user_id');
        $query = Commission::where('user_id', $userId);

        if ($request->has('start_date')) {
            $query->whereDate('created_at', '>=', $request->start_date);
        }
        if ($request->has('end_date')) {
            $query->whereDate('created_at', '<=', $request->end_date);
        }

        $commissions = $query->get();

        return response()->json([
            'success' => true,
            'user_id' => $userId,
            'total_commission' => $commissions->sum('commission_amount'),
            'count' => $commissions->count(),
            'commissions' => $commissions,
        ])->header('Cache-Control', 'no-cache, no-store, must-revalidate')
            ->header('Pragma', 'no-cache')
            ->header('Expires', '0');
    }

    public function topEarners(Request $request)
    {
        $limit = $request->query('limit', 10);

        $topEarners = Commission::with('user')
            ->selectRaw('user_id, SUM(commission_amount) as total_commission, COUNT(*) as total_completions')
            ->groupBy('user_id')
            ->orderByDesc('total_commission')
            ->limit($limit)
            ->get();

        return response()->json([
            'success' => true,
            'data' => $topEarners,
        ])->header('Cache-Control', 'no-cache, no-store, must-revalidate')
            ->header('Pragma', 'no-cache')
            ->header('Expires', '0');
    }
}
