<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;

class AuthController extends Controller
{
    /**
     * Handle user login
     */
    public function login(Request $request)
    {
        $request->validate([
            'code' => 'required|string',
            'password' => 'required|string',
        ]);

        // Check for super admin
        $superAdminCode = config('bbp.super_admin.code');
        $superAdminHash = config('bbp.super_admin.password_hash');

        if ($request->code === $superAdminCode && Hash::check($request->password, $superAdminHash)) {
            return response()->json([
                'success' => true,
                'token' => Str::random(32),
                'user' => [
                    'user_id' => 1,
                    'name' => 'Super Admin',
                    'role' => 'admin',
                    'designation' => 'Owner',
                    'code' => $superAdminCode,
                    'basic_pay' => 13000.00,
                ],
            ]);
        }

        // Check regular users
        $user = User::where('code', $request->code)
            ->whereIn('role', ['admin', 'employee'])
            ->first();

        if (!$user || !Hash::check($request->password, $user->password)) {
            throw ValidationException::withMessages([
                'code' => ['Invalid credentials'],
            ]);
        }

        return response()->json([
            'success' => true,
            'token' => Str::random(32),
            'user' => [
                'user_id' => $user->user_id,
                'name' => $user->name,
                'role' => $user->role,
                'designation' => $user->designation,
                'code' => $user->code,
                'basic_pay' => $user->basic_pay,
            ],
        ]);
    }

    /**
     * Handle user registration
     */
    public function register(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'role' => 'required|in:admin,employee',
            'designation' => 'nullable|string|max:255',
            'code' => 'required|string|unique:users,code',
            'basic_pay' => 'nullable|numeric|min:0',
            'password' => 'required|string|min:4',
        ]);

        $user = User::create([
            'name' => $validated['name'],
            'role' => $validated['role'],
            'designation' => $validated['designation'] ?? null,
            'code' => $validated['code'],
            'basic_pay' => $validated['basic_pay'] ?? 0,
            'password' => $validated['password'], // Will be auto-hashed by model
        ]);

        return response()->json([
            'success' => true,
            'message' => 'User registered successfully',
            'user' => [
                'user_id' => $user->user_id,
                'name' => $user->name,
                'role' => $user->role,
                'designation' => $user->designation,
                'code' => $user->code,
                'basic_pay' => $user->basic_pay,
            ],
        ], 201);
    }
}
