<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Service;
use App\Models\Completion;
use App\Models\Commission;
use App\Models\User;

class TestObservers extends Command
{
    protected $signature = 'observers:test';
    protected $description = 'Test that all observers are working correctly';

    public function handle(): int
    {
        $this->info('🧪 Testing Model Observers...');
        $this->newLine();

        // Test 1: Commission calculation
        $this->info('Test 1: Commission Auto-Calculation');
        $this->testCommissionCalculation();
        $this->newLine();

        // Test 2: Service commission update
        $this->info('Test 2: Service Rate Update (Recalculation)');
        $this->testServiceRateUpdate();
        $this->newLine();

        // Test 3: Timezone adjustments
        $this->info('Test 3: Timezone Adjustments');
        $this->testTimezoneAdjustments();
        $this->newLine();

        $this->info('✅ All observer tests completed!');
        return self::SUCCESS;
    }

    private function testCommissionCalculation(): void
    {
        try {
            // Get or create test data
            $service = Service::first();
            $user = User::first();

            if (!$service || !$user) {
                $this->warn('⚠️  No service or user found. Please seed database first.');
                return;
            }

            // Create a completion
            $completion = Completion::create([
                'service_id' => $service->service_id,
                'user_id' => $user->user_id,
                'charged' => 100.00,
                'date' => now(),
                'approved' => true,
            ]);

            // Create commission
            $commission = Commission::create([
                'completions_id' => $completion->completions_id,
            ]);

            // Refresh to get latest data
            $commission->refresh();

            // Expected calculation
            $expectedAmount = (100.00 * $service->commission) / 100;

            $this->line("   Service: {$service->service_name}");
            $this->line("   Commission Rate: {$service->commission}%");
            $this->line("   Charged: $100.00");
            $this->line("   Expected Amount: $" . number_format($expectedAmount, 2));
            $this->line("   Calculated Amount: $" . number_format($commission->commission_amount, 2));
            $this->line("   User ID Set: " . ($commission->user_id == $user->user_id ? '✓' : '✗'));

            if (abs($commission->commission_amount - $expectedAmount) < 0.01) {
                $this->info('   ✓ Commission calculated correctly!');
            } else {
                $this->error('   ✗ Commission calculation failed!');
            }

            // Cleanup
            $commission->delete();
            $completion->delete();
        } catch (\Exception $e) {
            $this->error('   ✗ Error: ' . $e->getMessage());
        }
    }

    private function testServiceRateUpdate(): void
    {
        try {
            $service = Service::first();
            $user = User::first();

            if (!$service || !$user) {
                $this->warn('⚠️  No service or user found.');
                return;
            }

            // Create completion and commission
            $completion = Completion::create([
                'service_id' => $service->service_id,
                'user_id' => $user->user_id,
                'charged' => 200.00,
                'date' => now(),
                'approved' => true,
            ]);

            $commission = Commission::create([
                'completions_id' => $completion->completions_id,
            ]);

            $oldAmount = $commission->fresh()->commission_amount;
            $this->line("   Initial Commission: $" . number_format($oldAmount, 2));

            // Update service rate
            $oldRate = $service->commission;
            $newRate = $oldRate + 5;
            $service->commission = $newRate;
            $service->save();

            // Check if commission was recalculated
            $commission->refresh();
            $newAmount = $commission->commission_amount;
            $expectedAmount = (200.00 * $newRate) / 100;

            $this->line("   Old Rate: {$oldRate}%");
            $this->line("   New Rate: {$newRate}%");
            $this->line("   New Commission: $" . number_format($newAmount, 2));
            $this->line("   Expected: $" . number_format($expectedAmount, 2));

            if (abs($newAmount - $expectedAmount) < 0.01) {
                $this->info('   ✓ Commission recalculated correctly!');
            } else {
                $this->error('   ✗ Recalculation failed!');
            }

            // Restore original rate
            $service->commission = $oldRate;
            $service->save();

            // Cleanup
            $commission->delete();
            $completion->delete();
        } catch (\Exception $e) {
            $this->error('   ✗ Error: ' . $e->getMessage());
        }
    }

    private function testTimezoneAdjustments(): void
    {
        try {
            $service = Service::first();
            $user = User::first();

            if (!$service || !$user) {
                $this->warn('⚠️  No service or user found.');
                return;
            }

            $beforeTime = now();

            $completion = Completion::create([
                'service_id' => $service->service_id,
                'user_id' => $user->user_id,
                'charged' => 50.00,
                'date' => $beforeTime,
                'approved' => true,
            ]);

            $completion->refresh();
            $adjustedTime = $completion->created_at;

            // Check if 7 hours were added (or close to it)
            $hoursDiff = $beforeTime->diffInHours($adjustedTime, false);

            $this->line("   Original Time: " . $beforeTime->format('Y-m-d H:i:s'));
            $this->line("   Adjusted Time: " . $adjustedTime->format('Y-m-d H:i:s'));
            $this->line("   Hours Difference: " . $hoursDiff);

            if (abs($hoursDiff - 7) <= 1) {
                $this->info('   ✓ Timezone adjustment applied!');
            } else {
                $this->warn('   ⚠️  Timezone adjustment may not be working as expected');
            }

            // Cleanup
            $completion->delete();
        } catch (\Exception $e) {
            $this->error('   ✗ Error: ' . $e->getMessage());
        }
    }
}
