<?php

namespace App\Console\Commands;

use App\Models\Completion;
use App\Models\Commission;
use App\Models\Service;
use App\Models\User;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class TestCompletionChanges extends Command
{
    protected $signature = 'test:completion-changes';
    protected $description = 'Test that commission table responds to ANY completion changes';

    public function handle()
    {
        $this->info('🧪 Testing Completion Observer - All Change Scenarios');
        $this->newLine();

        DB::beginTransaction();

        try {
            // Setup test data
            $user = User::first();
            $service = Service::first();

            if (!$user || !$service) {
                $this->error('❌ Need at least one User and one Service in database');
                DB::rollBack();
                return 1;
            }

            // TEST 1: Create approved completion -> Commission created
            $this->info('📝 Test 1: Create APPROVED completion -> Commission should be created');
            $completion1 = Completion::create([
                'service_id' => $service->service_id,
                'user_id' => $user->user_id,
                'charged' => 100,
                'approved' => true,
                'date' => now(),
            ]);
            $commission1 = Commission::where('completions_id', $completion1->completions_id)->first();
            $expectedAmount1 = (100 * $service->commission) / 100;

            if ($commission1 && abs($commission1->commission_amount - $expectedAmount1) < 0.01) {
                $this->info("   ✅ Commission created automatically: $$commission1->commission_amount");
            } else {
                $this->error("   ❌ Commission not created or wrong amount");
            }
            $this->newLine();

            // TEST 2: Change approved from true to false -> Commission deleted
            $this->info('📝 Test 2: Change approved TRUE → FALSE -> Commission should be DELETED');
            $this->info("   Commission ID before: {$commission1->commission_id}");
            $completion1->update(['approved' => false]);
            $commission1After = Commission::where('completions_id', $completion1->completions_id)->first();

            if (!$commission1After) {
                $this->info('   ✅ Commission deleted when unapproved');
            } else {
                $this->error('   ❌ Commission still exists after unapproval');
            }
            $this->newLine();

            // TEST 3: Change approved from false to true -> Commission created
            $this->info('📝 Test 3: Change approved FALSE → TRUE -> Commission should be CREATED');
            $completion1->update(['approved' => true]);
            $commission1New = Commission::where('completions_id', $completion1->completions_id)->first();

            if ($commission1New) {
                $this->info("   ✅ Commission re-created: ID {$commission1New->commission_id}, Amount: $" . $commission1New->commission_amount);
            } else {
                $this->error('   ❌ Commission not created when re-approved');
            }
            $this->newLine();

            // TEST 4: Update charged amount while approved -> Commission updated
            $this->info('📝 Test 4: Change charged amount (100 → 200) -> Commission should UPDATE');
            $oldAmount = $commission1New->commission_amount;
            $this->info("   Old commission: $$oldAmount");

            $completion1->update(['charged' => 200]);
            $commission1Updated = Commission::where('completions_id', $completion1->completions_id)->first();
            $expectedAmount4 = (200 * $service->commission) / 100;

            if ($commission1Updated && abs($commission1Updated->commission_amount - $expectedAmount4) < 0.01) {
                $this->info("   ✅ Commission updated to: $" . $commission1Updated->commission_amount);
            } else {
                $this->error('   ❌ Commission not updated correctly');
            }
            $this->newLine();

            // TEST 5: Change service while approved -> Commission recalculated
            if (Service::count() > 1) {
                $this->info('📝 Test 5: Change service -> Commission should RECALCULATE');
                $service2 = Service::where('service_id', '!=', $service->service_id)->first();
                $oldServiceRate = $service->commission;
                $newServiceRate = $service2->commission;
                $this->info("   Service commission rate: $oldServiceRate% → $newServiceRate%");

                $completion1->update(['service_id' => $service2->service_id]);
                $commission1AfterService = Commission::where('completions_id', $completion1->completions_id)->first();
                $expectedAmount5 = (200 * $service2->commission) / 100;

                if ($commission1AfterService && abs($commission1AfterService->commission_amount - $expectedAmount5) < 0.01) {
                    $this->info("   ✅ Commission recalculated: $" . $commission1AfterService->commission_amount);
                } else {
                    $this->error('   ❌ Commission not recalculated for new service');
                }
                $this->newLine();
            }

            // TEST 6: Delete completion -> Commission deleted
            $this->info('📝 Test 6: Delete completion -> Commission should be DELETED');
            $commissionIdBefore = Commission::where('completions_id', $completion1->completions_id)->value('commission_id');
            $this->info("   Commission ID before delete: $commissionIdBefore");

            $completion1->delete();
            $commissionAfterDelete = Commission::find($commissionIdBefore);

            if (!$commissionAfterDelete) {
                $this->info('   ✅ Commission deleted with completion');
            } else {
                $this->error('   ❌ Commission still exists after completion deleted');
            }
            $this->newLine();

            // TEST 7: Create unapproved completion -> No commission
            $this->info('📝 Test 7: Create UNAPPROVED completion -> Commission should NOT exist');
            $completion2 = Completion::create([
                'service_id' => $service->service_id,
                'user_id' => $user->user_id,
                'charged' => 150,
                'approved' => false,
                'date' => now(),
            ]);
            $commission2 = Commission::where('completions_id', $completion2->completions_id)->first();

            if (!$commission2) {
                $this->info('   ✅ No commission created for unapproved completion');
            } else {
                $this->error('   ❌ Commission created for unapproved completion (should not happen)');
            }
            $this->newLine();

            DB::rollBack();
            $this->info('✅ All tests completed! (Changes rolled back)');
            $this->newLine();
            $this->info('📊 Summary:');
            $this->info('   - Observer monitors ALL completion changes');
            $this->info('   - Approved TRUE: Commission created/updated');
            $this->info('   - Approved FALSE: Commission deleted');
            $this->info('   - Charged/Service changes: Commission recalculated');
            $this->info('   - Completion deleted: Commission deleted');

            return 0;
        } catch (\Exception $e) {
            DB::rollBack();
            $this->error('❌ Test failed: ' . $e->getMessage());
            return 1;
        }
    }
}
